/*-----------------------------------------------------------------------------

   QUASAR - q-gram Alignment based on Suffix ARrays

   Copyright (C) 1998 Stefan Burkhardt
   Author: Stefan Burkhardt <stburk@mpi-sb.mpg.de>
   This file is part of the QUASAR package.

   QUASAR is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   QUASAR is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the QUASAR package; see the file copying.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  or contact the author. 

-------------------------------------------------------------------------------

  block array module
  
  $File$
  $Revision: 1.3 $
  $Date: Wed, 29 Mar 2000 11:07:45 +0200 $

-----------------------------------------------------------------------------*/

#ifndef _Q_BLOCKS_H
#define _Q_BLOCKS_H

#include "q_misc.h"
#include "q_index.h"
#include "q_hood.h"
#include "q_options.h"
#include "q_sset.h"
#include "q_defines.h"

typedef struct q_blocks{
  int		n;
  int		b;
  int		t;
  int		*ba;
} Q_Blocks;

Q_Blocks *NewBlocks();
/*  Creates and initializes a block array */

void DeleteBlocks(Q_Blocks *blocks);
/* Frees all memory allocated for blocks */

Q_Blocks *InitBlocks(int n, int b, int t);
/*  Initialize a block array for a database with n letters, a
    block size of b and a block threshold of t, return a pointer to it.
    At the end of the block array, 3 sentinel entries (0, 0, 1) are set
    for use in GetFullBlocks 
    a block array covers a db as follows:

db:  ------------------------------------
     0   2   4   6   8   ....           n-1 
ba:|---|---|---|---|---|---|---|---|---|---|
     |---|---|---|---|---|---|---|---|---|
       1   3   5   7   9   ...
*/

void ClearBlocks(Q_Blocks *blocks);
/*  Sets all block array counters to 0 */

void ResetBlocks(Q_Blocks *blocks);
/*  Sets all block array counters with value below t to 0 */

void AddHits(Q_Blocks *blocks, int n, int *hits);
/*  Adds n hits to block array */

void DelHits(Q_Blocks *blocks, int n, int *hits);
/*  Deletes n hits from block array in blocks with value below t */

void AddSequence(Q_SSet *sset, char *seq, Q_Blocks *blocks, Q_Index *index, LONG *qgrams, Q_Options *op);
/*  Adds the sequence whose qgrams are passed in qgrams to 
    blocks using a window length of op->w. the first entry of 
    qgrams contains the number of qgrams. index is required to 
    access the hitlists for the qgrams if op->filter_mode = 0, 
    the standard exact q-gram filtration is applied, if 
    op->filter_mode = 1, the new 1-neighbourhood q-gram 
    filtration is used */

int *GetFullBlocks(Q_Blocks *blocks);
/*  Returns an integer array fb with the following structure:
    fb[0] 2 * number of interesting zones (number of used array cells)
    fb[1] - fb[fb[0]] pairs of integers delimiting interesting zones */

void PrintBlocksInfo(Q_Blocks *blocks);
/*  Prints block array information */

void PrintBlocks(Q_Blocks *blocks);
/*  Prints complete block array */

void PrintBlocksStatistics(Q_Blocks *blocks);
/*  Prints block array statistics */

#endif	/* _Q_BLOCKS_H */
